/*
 *
 * GWIC
 *
 * (c) Joonas Lehtinen (jole@jole.fi), TUCS, 1998
 *
 */

#include "gwic.h"
#include <math.h>

int parse_arguments(int argc, char **argv, parameters *p)
{
  /* Determine operation */
    argc--; argv++;
  if (argc) {
    if (**argv == 'c') p->operation = 'c';
    else if (**argv == 'd') p->operation = 'd';
    else return 1;
  } else return 1;
  argv++; argc--;

  /* Initialize default parameters */
  p->targetbpp = 1.0;
  p->transform = TR_DEFAULT;
  p->ccbsc = (int)(0.1 * 256.0);

  /* Parse options */
  while (argc && **argv == '-') {
    /* -bpp */
    if (!strcmp(*argv,"-bpp")) {
      argv++; argc--;
      if (argc) 
	p->targetbpp = atof(*argv);
      else return 1;
    }
    /* -cbw */
    else if (!strcmp(*argv,"-cbw")) {
      argv++; argc--;
      if (argc) 
	p->ccbsc = atof(*argv) * 256;
      else return 1;
    }
    else return 1;
    argv++; argc--;
  }

  /* Open the files */
  if (argc) {
    p->in = fopen(*argv,"rb");
    if (!p->in) return 1;
    argv++; argc--;
  } else p->in = stdin;

  if (argc) {
    p->out = fopen(*argv,"wb");
    if (!p->out) return 1;
    argv++; argc--;
  } else p->out = stdout;

  if (argc) return 1;

  return 0;
}

void print_usage_help(int argc, char **argv)
{
  fprintf(stderr, "GWIC - GNU Wavelet Image Codec (c) Joonas Lehtinen "
	  "(jole@jole.fi), TUCS\n");
  fprintf(stderr, "(color model YUV422 now used)\nUsage:\n");
  fprintf(stderr, 
	  "  gwic (c [-bpp bitsperpixel] [-cbw cbwidth] |d) [infile.wlt [outfile.pnm]]\n"
	  "  Targetbpp-rate defaults to 1.0\n"
	  "  cbwidth is colorbandwidth default 0.1,\n"
	  "  if input/ouput-file is not specified,\n"
	  "  stdin/stdout is assumed\n");
}

/* ---------------------------------------------------------------------- */
/*    M A I N                                                             */
/* ---------------------------------------------------------------------- */
main(int argc, char ** argv)
{
  float bpp;
  int w,h,ow,oh,maxc,levelsY, levelsUV;
  float *f, *f2;
  U32 *coeff, *coeff2, *coeff3;
  heap2d *heap,*heap2,*heap3;
  header head;
  parameters par;
  int i,j,k;
  FILE *df;

  /* Parse and check arguments */
  if (parse_arguments(argc,argv,&par)) {
    print_usage_help(argc,argv);
    return (1);
  }

  if (par.operation == 'c') {
    /* --- COMPRESS ---------------------------------------------- */

    load_pnm(par.in,&f,&w,&h,&ow,&oh,&maxc,&head.cm);
    head.magic = MAGIC; head.version = VERSION;
    head.iwidth = ow; head.iheight = oh;
    head.width = w; head.height = h;
    head.ccbsc = par.ccbsc;
    head.bpp[0] = maxc < 256 ? 8 : (maxc < 4096 ? 12 : 16); head.dpi = 0;
    head.transform = par.transform;
    switch (head.cm) {
    case CM_GRAY: 
      head.levelsY = levelsY = forward_transform(f,w,h,head.transform);
      coeff = convert_matrix_from_float(f,w*h);
      heap = gen_2d_heap(coeff, w, h, levelsY);  
      head.n[0] = heap->max;
      fwrite((void *)&head, 1, sizeof(header),par.out);
      start_coding(par.out,TOTALSTATES);
      zerotree_code(coeff,w,h,levelsY,
		    (int)(par.targetbpp * ((float)(w*h)/8.0)),
		    heap,ow*oh);
      break;
    case CM_GRAY_DEEPSPARC: 
      TERMINATE("Unsupported color model\n");
      break;
    case CM_RGB24: 
      head.bpp[1] = head.bpp[2] = head.bpp[0];
      if (head.bpp[0] != 8) {
	TERMINATE("Unsupported color model - RGB24 with BPP != 8\n");
      }
      convert_matrix_from_24bc_packed_to_yuv_planes(f,w*h);
      head.levelsY = levelsY = forward_transform(f,w,h,head.transform);
      head.levelsUV = levelsUV = forward_transform(f+w*h,w/2,h,head.transform);
      forward_transform(f+w*h+w*h/2,w/2,h,head.transform);
      coeff = convert_matrix_from_float(f,w*h*2);
      heap = gen_2d_heap(coeff, w, h, levelsY);  
      heap2 = gen_2d_heap(coeff+w*h, w/2, h, levelsUV);
      heap3 = gen_2d_heap(coeff+w*h+w*h/2, w/2, h, levelsUV);  
      head.n[0] = heap->max;
      head.n[1] = heap2->max;
      head.n[2] = heap3->max;
      fwrite((void *)&head, 1, sizeof(header),par.out);
      start_coding(par.out,TOTALSTATES);
      zerotree_code(coeff,w,h,levelsY,
//--		    (int)((par.targetbpp*w*h*32)/(256+2*head.ccbsc)),
//		    (int)((par.targetbpp*w*h*92)/800),
		    (int)((par.targetbpp*w*h*(256-2*head.ccbsc))/(256*8)),
		    heap,ow*oh);
      zerotree_code(coeff+w*h,w/2,h,levelsUV,
//--		    ((int)((256+head.ccbsc)*(par.targetbpp*w*h)/
//--		     (256+2*head.ccbsc)))>>3,
//		    (int)((par.targetbpp*w*h*96)/800),
		    (int)((par.targetbpp*w*h*(256-head.ccbsc))/(256*8)),
		    heap2,ow*oh);
      zerotree_code(coeff+w*h+w*h/2,w/2,h,levelsUV,
		    (int)((par.targetbpp*w*h)/8),
		    heap3,ow*oh);
      break;
    default:
      TERMINATE("Unsupported color model\n");
      break;
    }
    stop_coding(); 
  } else {
    /* --- DECOMPRESS -------------------------------------------- */

    fread((void *)&head, 1, sizeof(header),par.in);
    start_decoding(par.in,TOTALSTATES);
    switch (head.cm) {
    case CM_GRAY: 
      coeff = zerotree_decode(head.width,head.height,head.levelsY,head.n[0]);
      stop_decoding();
      f = convert_matrix_from_int(coeff,head.width*head.height); 
      inverse_transform(f,head.width,head.height,head.transform);
      break;
    case CM_GRAY_DEEPSPARC: 
      TERMINATE("Unsupported color model\n");
      break;
    case CM_RGB24: 
      coeff = zerotree_decode(head.width,head.height,head.levelsY,head.n[0]);
      coeff2 = zerotree_decode(head.width/2,head.height,head.levelsUV,head.n[1]);
      coeff3 = zerotree_decode(head.width/2,head.height,head.levelsUV,head.n[2]);
      stop_decoding();
      MALLOC(f2,sizeof(float)*3*head.width*head.height);
      j = head.width * head.height;

      f = convert_matrix_from_int( coeff, head.width * head.height);
      inverse_transform( f, head.width, head.height, head.transform);
      for(i=0; i<j; i++) *(f2+i) = *(f+i); free(f);

      f = convert_matrix_from_int(coeff2, head.width * head.height / 2); 
      inverse_transform(f, head.width / 2, head.height, head.transform);
      for(i=0; i<j/2; i++) *(f2+i+j) = *(f+i); free(f);

      f = convert_matrix_from_int( coeff3, head.width * head.height / 2); 
//      f = coeff2 + head.width * head.height / 2;
      inverse_transform( f, head.width / 2, head.height, head.transform);
      for(i=0; i<j/2; i++) *(f2+i+j+j/2) = *(f+i); free(f);

      f = f2;
      convert_matrix_from_to_yuv_planes_24bc_packed(f,j);
      break;
    default:
      TERMINATE("Unsupported color model\n");
      break;
    }
    save_pnm(par.out,f,head.width,head.height,head.iwidth,head.iheight,
	     (1<<(head.bpp[0]))-1,head.cm);
  }

  return 0;
}





