/*
 *
 * GWIC
 *
 * (c) Joonas Lehtinen (jole@jole.fi), TUCS, 1998
 *
 */

/* This file contains all the prototypes for GWIC compession package */

#ifndef dlwic_h
#define dlwic_h

#include <stdio.h>
#include <stdlib.h>


/* --------------------------- */
/* --- General definitions --- */
/* --------------------------- */

/* Bitplanes: n \in [32,1] */
#define MASK 0x7fffffff
#define SIGN 0x80000000

/* We went to handle raw PGM:s using intel byte-order */
#define INTELPGMBYTEORDER 1

/* Initialize acrhitecture specific types */
typedef  unsigned short    U16;
typedef  unsigned long     U32;
typedef  unsigned char     BYTE;

/* Structure for two dimensional heap */
typedef struct {
  BYTE *heap;
  BYTE **lines;
  int levels;
  int width;
  int height;
  int max;
} heap2d;

/* Parameters parsing structure */
typedef struct {
  FILE *in;
  FILE *out;
  char operation;	/* either 'c' for compress or 'd' for decompress */
  float targetbpp;	/* Target byterate */
  BYTE transform;
  U32 ccbsc;		/* Crominace Channel BPP Scaling Coefficient
			   ccbpp = (bbp * ccbsc) >> 8 */
} parameters;


/* Define identification for this version */
#define MAGIC 821787163
#define VERSION 1

/* Define different wavelet transforms */
#define TOTAL_TRANSFORMS  2   /* Number of supported transforms */
#define TR_DAUB2          0   /* Daubechies transform */
#define TR_B97            1   /* Biorthogonal B97 transform with
			         circular vector boudaries */
static char *tr_names[] = {
  /* Names of the supported transforms */
  "daub2",
  "b97",
};
#define TR_DEFAULT TR_B97    /* Default transform used */

/* Define different color-models */
#define TOTAL_COLOR_MODELS 3   /* Number of different color-models supported */
#define CM_GRAY            0   /* Gray scale image 8 intel-byte-order */
#define CM_GRAY_DEEPSPARC  1   /* Gray scale image with >8 bpp and sparc
				  byte-order */
#define CM_RGB24           2   /* 24bit packed RGB color */

typedef struct {
  U32 magic;
  U16 version;            /* Version number */
  U16 iwidth,iheight;     /* Dimensions of the original image */
  U16 dpi;                /* Dots Per Inch, 0 if unspecified */
  U32 ccbsc;              /* Crominace Channel BPP Scaling Coefficient
			     ccbpp = (bbp * ccbsc) >> 8 */
  BYTE cm;                /* Color model used */
  BYTE transform;         /* Transform used */
  U16 width,height;       /* Dimensions of the coefficient matrix */
  BYTE levelsY;            /* Number of Y levels in pyramid composition */
  BYTE levelsUV;            /* Number of U&V levels in pyramid composition */
  BYTE bpp[4];            /* Bits per pixel on each color channel */
  BYTE n[4];              /* Starting scanning level on a color channel */
} header;

/* ----------------- */
/* --- Transform --- */
/* ----------------- */

int forward_transform(float *table, int width, int height, BYTE transform);
void inverse_transform(float *table, int width, int height, BYTE transform);

/* ------------------- */
/* --- Conversions --- */
/* ------------------- */

/* Convert and round float matrix to sign-int format */
U32 *convert_matrix_from_float(float *buf, int len);

/* Convert sing-int formatted matrix to floats */
float *convert_matrix_from_int(U32 *buf, int len);

void convert_matrix_from_24bc_packed_to_yuv_planes(float *buf, int len);
void convert_matrix_from_to_yuv_planes_24bc_packed(float *buf, int len);

/* --------------- */
/* --- 2D Heap --- */
/* --------------- */

/* Construct 2D significance heap */
heap2d *gen_2d_heap(U32 *buf, int width, int height, int levels);

/* ----------- */
/* --- PNM --- */
/* ----------- */

/* Load PGM file. Coverts dimensions to be more reasonable */
void load_pnm(FILE *in, float **pic, int *width, int *height, 
	      int *orig_width, int *orig_height, int *maxcolor, 
	      BYTE *colormode);

/* Save PGM file */
void save_pnm(FILE *out, float *pic, int width, int height, 
	      int orig_width, int orig_height, int maxcolor, 
	      BYTE colormode);

/* ------------- */
/* --- Coder --- */
/* ------------- */

void start_coding(FILE *f, int maxstates);
void code_bit(int bit, int state);
void code_buf(void *buf, int size, int state);
void stop_coding();
void start_decoding(FILE *f, int maxstates);
int decode_bit(int state);
void decode_buf(void *buf, int size, int state);
void stop_decoding();
int get_sent_bytes();

/* STATES */
#define SSIGN 0
#define SCONT 1
#define SBIT  2
#define SNEWSIG 3  /* Size 32 */
#define SORISIGH 35 /* Size 32 */
#define SORISIGD 67 /* Size 32 */
#define SORISIGV 99 /* Size 32 */
#define STOPLEVEL 131

#define TOTALSTATES 132

/* ---------------- */
/* --- Zerotree --- */
/* ---------------- */

void zerotree_code(U32 *coeff, int width, int height, int levels, 
		   int maxbytes, heap2d *heap, 
		   int real_image_area);
U32 *zerotree_decode(int width, int height, int levels, int n);

/* --------------------------------------------------- */
/* --- Error handling and memory allocation macros --- */
/* --------------------------------------------------- */

#define TERMINATE(msg) \
  {fprintf(stderr,"ERROR(%i @ %s): %s\n",__LINE__,__FILE__,msg); \
  exit(1);}

#define WARNING(msg) \
  fprintf(stderr,"WARNING(%i @ %s): %s\n",__LINE__,__FILE__,msg)

#define MALLOC(pointer,len) \
  if (!((void *) pointer = (void *)malloc(len))) { \
    TERMINATE("Memory allocation failed");}

#endif
